# SportPulse Widget API

A read-only REST API for retrieving sports data, leagues, widget types, and embed-ready templates for integrating sports widgets into third-party applications.

## Tech Stack

- Node.js + TypeScript
- Express.js 5.x
- MongoDB / Mongoose (optional - can use JSON file store)
- Zod validation
- Docker Compose (for local MongoDB)

## Quick Start

### Prerequisites

- Node.js 18+
- npm or yarn
- Docker (optional, for MongoDB)

### Installation

```bash
# Install dependencies
npm install

# Copy environment file
cp .env.example .env
```

### Environment Configuration

```bash
# .env
MONGODB_URI=mongodb://localhost:27018/sportpulse
API_PORT=3001
NODE_ENV=development

# Use JSON file as data store (no MongoDB required)
USE_JSON_STORE=true
```

Set `USE_JSON_STORE=true` to run without MongoDB (uses in-memory data from seed file).

### Running the Server

```bash
# Development mode (with hot reload)
npm run dev

# Production build
npm run build
npm start
```

The API will be available at `http://localhost:3001`

### Using MongoDB (Optional)

```bash
# Start MongoDB container
docker-compose up -d

# Seed the database
npm run seed

# Update .env
USE_JSON_STORE=false
```

## API Documentation

- **Interactive Docs**: http://localhost:3001/docs (ReDoc)
- **OpenAPI Spec (JSON)**: http://localhost:3001/openapi.json
- **OpenAPI Spec (YAML)**: http://localhost:3001/openapi.yaml

## Authentication

All API endpoints require an `X-API-Key` header.

**Development key**: `test-api-key-for-development`

```bash
curl -H "X-API-Key: test-api-key-for-development" http://localhost:3001/api/v1/sports
```

### Generating API Keys

```bash
npm run generate-key
```

## API Endpoints

Base URL: `/api/v1`

| Endpoint | Description |
|----------|-------------|
| `GET /sports` | List all sports |
| `GET /sports/:sportId/leagues` | Get leagues for a sport |
| `GET /leagues/:leagueId/participants` | Get participants in a league |
| `GET /leagues/:leagueId/widget-types` | Get widget types for a league |
| `GET /templates` | Get templates with embed codes |

### Query Parameters

| Endpoint | Parameters |
|----------|------------|
| `/sports` | `isActive` |
| `/sports/:sportId/leagues` | `isActive` |
| `/leagues/:leagueId/participants` | `type` (TEAM/DRIVER/PLAYER), `isActive` |
| `/leagues/:leagueId/widget-types` | `isActive` |
| `/templates` | `sportId`, `leagueId`, `widgetTypeId`, `participantId`, `featured`, `isActive` |

## Example Requests

### Get all sports
```bash
curl -H "X-API-Key: test-api-key-for-development" \
  http://localhost:3001/api/v1/sports
```

### Get leagues for a sport
```bash
curl -H "X-API-Key: test-api-key-for-development" \
  http://localhost:3001/api/v1/sports/3/leagues
```

### Get widget types for a league
```bash
curl -H "X-API-Key: test-api-key-for-development" \
  http://localhost:3001/api/v1/leagues/7/widget-types
```

### Get templates with embed codes
```bash
curl -H "X-API-Key: test-api-key-for-development" \
  "http://localhost:3001/api/v1/templates?sportId=3&featured=true"
```

## Project Structure

```
├── src/
│   ├── index.ts              # Entry point
│   ├── app.ts                # Express app setup
│   ├── config/               # Environment & database config
│   ├── middleware/           # Auth, error handling, logging
│   ├── models/               # Mongoose schemas
│   ├── routes/               # API route definitions
│   ├── controllers/          # Request handlers
│   ├── services/             # Business logic
│   ├── types/                # TypeScript interfaces
│   ├── validation/           # Zod schemas
│   ├── utils/                # Utilities
│   └── data/                 # JSON data store
├── scripts/                  # CLI utilities
├── sportpulse_widgets_seed.json  # Seed data
├── openapi.yaml              # API specification
└── docker-compose.yml        # MongoDB container
```

## Scripts

| Command | Description |
|---------|-------------|
| `npm run dev` | Start dev server with hot reload |
| `npm run build` | Compile TypeScript |
| `npm start` | Run production build |
| `npm run seed` | Load seed data into MongoDB |
| `npm run generate-key` | Generate new API key |
| `npm test` | Run tests |

## License

ISC
